<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class CopyTrade extends Model
{
    protected $fillable = [
        'follower_id',
        'trader_id',
        'amount_per_trade',
        'duration',
        'status',
        'total_copied_amount',
        'total_profit_loss',
        'copied_trades_count',
        'last_copy_at'
    ];

    protected $casts = [
        'amount_per_trade' => 'decimal:8',
        'total_copied_amount' => 'decimal:8',
        'total_profit_loss' => 'decimal:8',
        'last_copy_at' => 'datetime'
    ];

    // Relationships
    public function follower(): BelongsTo
    {
        return $this->belongsTo(User::class, 'follower_id');
    }

    public function trader(): BelongsTo
    {
        return $this->belongsTo(User::class, 'trader_id');
    }

    public function copiedTrades(): HasMany
    {
        return $this->hasMany(CopiedTrade::class);
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    // Methods
    public function pause()
    {
        $this->update(['status' => 'paused']);
    }

    public function resume()
    {
        $this->update(['status' => 'active']);
    }

    public function stop()
    {
        $this->update(['status' => 'stopped']);
    }

    public function updateStats()
    {
        $this->update([
            'total_copied_amount' => $this->copiedTrades()->sum('amount'),
            'total_profit_loss' => $this->copiedTrades()
                ->whereHas('copiedTrade', function ($query) {
                    $query->where('status', 'completed');
                })
                ->sum('profit_amount'),
            'copied_trades_count' => $this->copiedTrades()->count(),
            'last_copy_at' => $this->copiedTrades()->latest()->first()?->created_at
        ]);
    }
}