<?php

namespace App\Livewire\Admin\Kyc;

use App\Models\Kyc;
use App\Models\Settings;
use App\Notifications\User\KycApprovedNotification;
use App\Notifications\User\KycFailedNotification;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Storage;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\Layout;
use Livewire\Component;

#[Layout('layouts.admin')]
class SingleApplication extends Component
{
    use LivewireAlert;

    public Kyc $kyc;
    public $reason;
    public $action = 'verified';
    public $status;

    public function mount(Kyc $kyc)
    {
        $this->kyc = $kyc;
        $this->status = $kyc->status;
    }

    public function render()
    {
        return view('livewire.admin.kyc.single-application');
    }

    public function submit()
    {
        Gate::authorize('process kyc applications');

        $settings = Cache::get('site_settings');

        if ($this->action === 'verified') {
            $this->kyc->user->update(['account_verify' => $this->action]);
            $this->kyc->update(['status' => $this->action]);
            if ($settings->send_kyc_status_email) {
                $message = 'Your Kyc has been verified. You can now place a withdrawal request.';
                try {
                    $this->kyc->user->notify(new KycApprovedNotification(kyc: $this->kyc, message: $message));
                } catch (\Throwable $e) {
                    \Log::error('KYC approve notify failed', [
                        'kyc_id' => $this->kyc->id,
                        'user_id' => $this->kyc->user_id,
                        'error' => $e->getMessage(),
                    ]);
                }
            }
            $this->status = $this->kyc->status;
            $this->flash(message: 'Kyc Verified Successfully.', redirect: route('admin.kycApplications'));
        }

        if ($this->action === 'reject') {
            // Delete files from assets directory
            if (!empty($this->kyc->frontimg)) {
                $frontPath = public_path($this->kyc->frontimg);
                if (file_exists($frontPath)) {
                    unlink($frontPath);
                }
            }
            if (!empty($this->kyc->backimg)) {
                $backPath = public_path($this->kyc->backimg);
                if (file_exists($backPath)) {
                    unlink($backPath);
                }
            }

            if ($settings->send_kyc_status_email) {
                $message = 'Your Kyc has been rejected. Please check your email for more details.';
                try {
                    $this->kyc->user->notify(new KycFailedNotification(kyc: $this->kyc, message: $message, reason: $this->reason));
                } catch (\Throwable $e) {
                    \Log::error('KYC reject notify failed', [
                        'kyc_id' => $this->kyc->id,
                        'user_id' => $this->kyc->user_id,
                        'error' => $e->getMessage(),
                    ]);
                }
            }

            $this->kyc->user->update(['account_verify' => $this->action]);

            $this->kyc->delete();

            $this->flash(message: 'Kyc Rejected Successfully.', redirect: route('admin.kycApplications'));
        }
    }
}