<?php

namespace App\Console\Commands;

use App\Models\CryptoAsset;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class SyncCryptoPrices extends Command
{
    protected $signature = 'crypto:sync {--initial : Initial sync with logo fetch}';
    protected $description = 'Sync cryptocurrency prices and data from CoinGecko';

    private $excludedCoins = ['wrapped-steth', 'weth', 'steth', 'wrapped-bitcoin', 'wbtc'];

    public function handle()
    {
        $isInitial = $this->option('initial');
        
        $this->info('Starting crypto price sync...');
        
        try {
            $response = Http::timeout(15)->get('https://api.coingecko.com/api/v3/coins/markets', [
                'vs_currency' => 'usd',
                'order' => 'market_cap_desc',
                'per_page' => 250,
                'page' => 1,
                'sparkline' => false,
                'price_change_percentage' => '24h'
            ]);

            if (!$response->successful()) {
                throw new \Exception('Failed to fetch data from CoinGecko: ' . $response->status());
            }

            $coins = $response->json();
            $processed = 0;
            $skipped = 0;

            foreach ($coins as $coin) {
                // Skip wrapped tokens and excluded coins
                if ($this->shouldSkipCoin($coin)) {
                    $skipped++;
                    continue;
                }

                $data = [
                    'symbol' => strtoupper($coin['symbol']),
                    'name' => $coin['name'],
                    'coingecko_id' => $coin['id'],
                    'current_price' => $coin['current_price'] ?? 0,
                    'price_change_24h' => $coin['price_change_24h'] ?? 0,
                    'price_change_percentage_24h' => $coin['price_change_percentage_24h'] ?? 0,
                    'market_cap' => $coin['market_cap'] ?? 0,
                    'total_volume' => $coin['total_volume'] ?? 0,
                    'high_24h' => $coin['high_24h'] ?? 0,
                    'low_24h' => $coin['low_24h'] ?? 0,
                    'market_cap_rank' => $coin['market_cap_rank'],
                    'last_updated' => now(),
                ];

                // Only include logo on initial sync or if not set
                if ($isInitial || !CryptoAsset::where('symbol', $data['symbol'])->exists()) {
                    $data['logo_url'] = $coin['image'] ?? null;
                }

                CryptoAsset::updateOrCreate(
                    ['symbol' => $data['symbol']],
                    $data
                );

                $processed++;
            }

            $this->info("✓ Successfully synced {$processed} cryptocurrencies");
            $this->comment("Skipped {$skipped} wrapped/excluded tokens");
            
            // Log success
            Log::info('Crypto prices synced', [
                'processed' => $processed,
                'skipped' => $skipped,
                'is_initial' => $isInitial
            ]);

            return 0;

        } catch (\Exception $e) {
            $this->error('Failed to sync crypto prices: ' . $e->getMessage());
            Log::error('Crypto sync failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return 1;
        }
    }

    private function shouldSkipCoin(array $coin): bool
    {
        // Skip if in excluded list
        if (in_array($coin['id'], $this->excludedCoins)) {
            return true;
        }

        // Skip wrapped tokens
        if (str_starts_with(strtolower($coin['id']), 'wrapped-') ||
            str_starts_with(strtolower($coin['symbol']), 'w')) {
            return true;
        }

        // Skip USDT and stablecoins
        if (in_array(strtoupper($coin['symbol']), ['USDT', 'USDC', 'BUSD', 'DAI'])) {
            return true;
        }

        return false;
    }
}
